const SERVER_URL = "https://id-tunnel.duckdns.org";

document.addEventListener("DOMContentLoaded", async () => {
    const input = document.getElementById("apiKeyInput");
    const saveBtn = document.getElementById("saveKey");
    const clearBtn = document.getElementById("clearKey");
    const checkBtn = document.getElementById("checkServer");
    const refreshBtn = document.getElementById("refreshSettings");
    const enabledToggle = document.getElementById("enabledToggle");
    const rescanBtn = document.getElementById("rescanPage");
    const keyStatusEl = document.getElementById("keyStatus");
    const serverStatusEl = document.getElementById("serverStatus");
    const settingsEl = document.getElementById("currentSettings");
    const parsersListEl = document.getElementById("parsersList");
    const parsersCountEl = document.getElementById("parsersCount");

    let currentParsers = [];
    let currentSite = "";

    // Загрузка текущего ключа и настроек
    await loadInitialData();

    // Загрузка начальных данных
    async function loadInitialData() {
        try {
            // Загружаем API ключ
            const result = await new Promise(resolve => {
                chrome.storage.local.get(["apiKey", "my_api_key", "enabled"], resolve);
            });
            
            input.value = result.apiKey || result.my_api_key || "";
            enabledToggle.checked = result.enabled !== false;
            
            // Загружаем информацию о парсерах и настройках
            await loadParsersAndSettings();
            
            if (input.value) {
                showStatus(keyStatusEl, "✅ Ключ загружен", "success");
            }
        } catch (error) {
            console.error("Ошибка загрузки данных:", error);
            showStatus(keyStatusEl, "❌ Ошибка загрузки", "error");
        }
    }

    // Загрузка парсеров и настроек
    async function loadParsersAndSettings() {
        try {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (!tab || !tab.id) {
                throw new Error("Не удалось найти активную вкладку");
            }

            // Получаем текущий сайт
            const url = new URL(tab.url);
            currentSite = url.hostname.replace(/^www\./, '');

            // Пытаемся получить данные из content script
            try {
                const response = await chrome.tabs.sendMessage(tab.id, { 
                    action: "getParsersInfo" 
                });

                if (response && response.success) {
                    currentParsers = response.parsers || [];
                    displayParsers(currentParsers);
                    displayCurrentSettings(response.settings, response.site);
                    parsersCountEl.textContent = currentParsers.length.toString();
                    return;
                }
            } catch (error) {
                console.log("Content script не ответил, загружаем напрямую с сервера");
            }

            // Fallback: прямая загрузка с сервера
            await loadFromServerDirectly();
            
        } catch (error) {
            console.error("Ошибка загрузки парсеров и настроек:", error);
            showFallbackState();
        }
    }

    // Прямая загрузка с сервера
    async function loadFromServerDirectly() {
        try {
            // Загружаем парсеры через новый эндпоинт экспорта
            const response = await fetch(`${SERVER_URL}/api/parsers/export?site=${encodeURIComponent(currentSite)}`);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}`);
            }

            const data = await response.json();
            
            if (data.success && data.parsers) {
                currentParsers = data.parsers;
                displayParsers(currentParsers);
                parsersCountEl.textContent = currentParsers.length.toString();
            } else {
                // Fallback на старый эндпоинт
                const fallbackResponse = await fetch(`${SERVER_URL}/api/parsers?site=${encodeURIComponent(currentSite)}`);
                if (fallbackResponse.ok) {
                    const parsers = await fallbackResponse.json();
                    currentParsers = parsers.filter(parser => parser.is_active !== false);
                    displayParsers(currentParsers);
                    parsersCountEl.textContent = currentParsers.length.toString();
                }
            }
            
            // Загружаем настройки сайта
            const settingsResponse = await fetch(`${SERVER_URL}/api/site-settings?site=${encodeURIComponent(currentSite)}`);
            if (settingsResponse.ok) {
                const settings = await settingsResponse.json();
                displayCurrentSettings(settings, currentSite);
            } else {
                displayDefaultSettings();
            }
            
        } catch (error) {
            console.error("Ошибка прямой загрузки:", error);
            showFallbackState();
        }
    }

    // Показать состояние fallback
    function showFallbackState() {
        parsersListEl.innerHTML = '<div class="no-parsers">❌ Ошибка загрузки парсеров</div>';
        parsersCountEl.textContent = '0';
        displayDefaultSettings();
    }

    // Показать настройки по умолчанию
    function displayDefaultSettings() {
        settingsEl.innerHTML = `
            <div class="setting-card">
                <div class="setting-header">
                    <div class="setting-name">Настройки не загружены</div>
                    <div class="setting-value">Обновите страницу или проверьте сервер</div>
                </div>
            </div>
        `;
    }

    // Функция отображения статуса
    function showStatus(element, message, type) {
        element.textContent = message;
        element.className = `status ${type}`;
        element.style.display = 'block';
        setTimeout(() => {
            element.style.display = 'none';
        }, 5000);
    }

    // Отображение списка парсеров
    function displayParsers(parsers) {
        if (!parsers || parsers.length === 0) {
            parsersListEl.innerHTML = '<div class="no-parsers">🎯 Нет активных парсеров для этого сайта</div>';
            return;
        }

        const html = parsers.map(parser => `
            <div class="parser-item">
                <div class="parser-name">${escapeHtml(parser.name)}</div>
                <div style="font-size: 11px; color: #6c757d; margin-top: 2px;">
                    Тип: ${parser.type || parser.pattern_type || 'auto'} • Сайт: ${parser.site}
                </div>
                ${parser.description ? `
                    <div style="font-size: 10px; color: #868e96; margin-top: 2px;">
                        ${escapeHtml(parser.description)}
                    </div>
                ` : ''}
                <div style="font-size: 9px; color: #adb5bd; margin-top: 2px; font-family: monospace; word-break: break-all;">
                    ${escapeHtml(parser.pattern).substring(0, 60)}${parser.pattern.length > 60 ? '...' : ''}
                </div>
            </div>
        `).join('');

        parsersListEl.innerHTML = html;
    }

    // Функция отображения текущих настроек
    function displayCurrentSettings(settings, site) {
        if (!settings) {
            displayDefaultSettings();
            return;
        }

        let html = `
            <div class="setting-card">
                <div class="setting-header">
                    <div class="setting-name">Текущий сайт</div>
                    <div class="setting-value">${site || 'Неизвестно'}</div>
                </div>
            </div>

            <div class="setting-card">
                <div class="setting-header">
                    <div class="setting-name">Минимальное количество</div>
                    <div class="setting-value">${settings.minDuplicates || 2}</div>
                </div>
                <div class="setting-status ${settings.useMinDuplicates ? 'status-enabled' : 'status-disabled'}">
                    ${settings.useMinDuplicates !== false ? '✅ Включено' : '❌ Выключено'}
                </div>
            </div>

            <div class="setting-card">
                <div class="setting-header">
                    <div class="setting-name">Минимальная дата</div>
                    <div class="setting-value">${new Date(settings.minDate || '2025-09-22').toLocaleDateString('ru-RU')}</div>
                </div>
                <div class="setting-status ${settings.useMinDate ? 'status-enabled' : 'status-disabled'}">
                    ${settings.useMinDate !== false ? '✅ Включено' : '❌ Выключено'}
                </div>
            </div>
        `;

        if (settings.siteGroups && settings.siteGroups.length > 0) {
            html += `
                <div class="setting-card">
                    <div class="setting-header">
                        <div class="setting-name">Группы сайтов</div>
                        <div class="setting-value">${settings.siteGroups.length}</div>
                    </div>
                    <div class="groups-list">
                        <span class="group-tag current-site">${site}</span>
                        ${settings.siteGroups.map(group => 
                            `<span class="group-tag">${escapeHtml(group)}</span>`
                        ).join('')}
                    </div>
                </div>
            `;
        }

        settingsEl.innerHTML = html;
    }

    // Экранирование HTML
    function escapeHtml(unsafe) {
        if (!unsafe) return '';
        return unsafe.toString()
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }

    // Сохранение ключа
    saveBtn.addEventListener("click", async () => {
        const key = input.value.trim();
        if (!key) {
            showStatus(keyStatusEl, "⚠️ Введите API-ключ", "warning");
            return;
        }

        try {
            await chrome.storage.local.set({ apiKey: key, my_api_key: key });
            showStatus(keyStatusEl, "✅ Ключ успешно сохранён", "success");
            
            // Анимация кнопки
            saveBtn.innerHTML = '<span>✅</span> Успешно';
            setTimeout(() => {
                saveBtn.innerHTML = '<span>💾</span> Сохранить';
            }, 2000);
            
        } catch (err) {
            showStatus(keyStatusEl, "❌ Ошибка при сохранении ключа", "error");
        }
    });

    // Очистка ключа
    clearBtn.addEventListener("click", async () => {
        if (!confirm("Вы уверены, что хотите удалить API-ключ?")) return;
        
        try {
            await chrome.storage.local.remove(["apiKey", "my_api_key"]);
            input.value = "";
            showStatus(keyStatusEl, "🗑️ Ключ удалён", "success");
            
            // Анимация кнопки
            clearBtn.innerHTML = '<span>✅</span> Удалено';
            setTimeout(() => {
                clearBtn.innerHTML = '<span>🗑️</span> Очистить';
            }, 2000);
            
        } catch (err) {
            showStatus(keyStatusEl, "❌ Ошибка при удалении ключа", "error");
        }
    });

    // Проверка соединения
    checkBtn.addEventListener("click", async () => {
        checkBtn.innerHTML = '<span>⏳</span> Проверка...';
        checkBtn.disabled = true;
        
        try {
            const response = await fetch(`${SERVER_URL}/api/ping`);
            
            if (response.ok) {
                showStatus(serverStatusEl, "✅ Сервер доступен", "success");
            } else {
                showStatus(serverStatusEl, `⚠️ Ошибка сервера (${response.status})`, "warning");
            }
        } catch {
            showStatus(serverStatusEl, "❌ Сервер недоступен", "error");
        } finally {
            checkBtn.innerHTML = '<span>🔄</span> Проверить соединение';
            checkBtn.disabled = false;
        }
    });

    // Обновление настроек и парсеров
    refreshBtn.addEventListener("click", async () => {
        refreshBtn.innerHTML = '<span>⏳</span> Обновление...';
        refreshBtn.disabled = true;
        
        try {
            // Обновляем настройки через content script
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (tab && tab.id) {
                await chrome.tabs.sendMessage(tab.id, { 
                    action: "refreshSettings" 
                });
            }
            
            // Перезагружаем данные
            await loadParsersAndSettings();
            showStatus(serverStatusEl, "✅ Данные обновлены", "success");
        } catch (error) {
            console.error("Ошибка обновления:", error);
            showStatus(serverStatusEl, "❌ Ошибка обновления", "error");
        } finally {
            refreshBtn.innerHTML = '<span>⚡</span> Обновить';
            refreshBtn.disabled = false;
        }
    });

    // Переключатель авто-сканирования
    enabledToggle.addEventListener("change", async () => {
        const enabled = enabledToggle.checked;
        await chrome.storage.local.set({ enabled });
        
        // Уведомляем content script
        try {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (tab && tab.id) {
                await chrome.tabs.sendMessage(tab.id, { 
                    action: "toggleAutoScan",
                    enabled: enabled
                });
            }
        } catch (error) {
            console.log("Не удалось уведомить content script:", error);
        }
        
        showStatus(serverStatusEl, 
            enabled ? "✅ Авто-сканирование включено" : "⏸️ Авто-сканирование выключено", 
            "info"
        );
    });

    // Кнопка пересканирования страницы
    rescanBtn.addEventListener('click', async () => {
        rescanBtn.innerHTML = '<span>⏳</span> Сканируем...';
        rescanBtn.disabled = true;
        
        try {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (tab && tab.id) {
                // Отправляем запрос на пересканирование в content script
                const response = await chrome.tabs.sendMessage(tab.id, { 
                    action: "rescanPage" 
                });
                
                if (response && response.success) {
                    showStatus(serverStatusEl, "✅ Страница пересканирована", "success");
                    
                    // Обновляем данные после пересканирования
                    setTimeout(async () => {
                        await loadParsersAndSettings();
                    }, 1000);
                } else {
                    throw new Error("Content script не ответил");
                }
            }
        } catch (error) {
            console.error("Ошибка пересканирования:", error);
            showStatus(serverStatusEl, "❌ Ошибка пересканирования", "error");
        } finally {
            rescanBtn.innerHTML = '<span>🔍</span> Пересканировать';
            rescanBtn.disabled = false;
        }
    });

    // Слушаем сообщения от content script
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
        if (request.action === 'parsersUpdated') {
            currentParsers = request.parsers || [];
            displayParsers(currentParsers);
            parsersCountEl.textContent = currentParsers.length.toString();
        }
        
        if (request.action === 'settingsUpdated') {
            displayCurrentSettings(request.settings, request.site);
        }
    });

    // Автофокус на поле ввода
    input.focus();
});